/*
 * gcc -Wall pi-monte-carlo.c -o pi-monte-carlo.c.exe -lm
 * clang -Wall pi-monte-carlo.c -o pi-monte-carlo.c.exe -lm
 * icc -Wall pi-monte-carlo.c -o pi-monte-carlo.c.exe
 * icx -Wall pi-monte-carlo.c -o pi-monte-carlo.c.exe
 */
#define _GNU_SOURCE

#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>

static inline double
get_time()
{
    struct timespec tp;
    int result = clock_gettime(CLOCK_MONOTONIC, &tp);

    if (result == -1) {
        fprintf(stderr,
                "ERROR: clock_gettime(CLOCK_MONOTONIC) failed: %d - %s\n",
                errno, strerror(errno));
        exit(EXIT_FAILURE);
    }

    return (double)tp.tv_sec + (double)tp.tv_nsec * 1e-9;
}


int main()
{
    unsigned int seed = 2;
    int count = 0;
    int n_repetitions = 342000000;
    double t_start = get_time();
    double x;
    double y;

    for (int i = 0; i < n_repetitions; ++i) {
        x = rand_r(&seed) / (double)RAND_MAX;
        y = rand_r(&seed) / (double)RAND_MAX;

        if (sqrt(x * x + y * y) < 1.0) {
            ++count;
        }
    }

    double pi = 4.0 * (double)count / n_repetitions;
    double duration = get_time() - t_start;

    printf("count: %d  repetitions: %d\n", count, n_repetitions);
    printf("perf: %.3lf Mit/sec, duration: %f s, accuracy: %.12e\n",
         (double)n_repetitions / duration / 1.0e6,
         duration,
         fabs(M_PI - pi) / M_PI);

    return 0;
}
