#define _GNU_SOURCE
#include <sched.h>
#include <stdio.h>
#include <unistd.h>

#ifdef __linux__
#include <sys/syscall.h>
#include <sys/types.h>
#define gettid() (int)syscall(SYS_gettid)
#endif

#ifdef _OPENMP
#include <omp.h>
#endif

#define MAX_NUM_THREADS 256
#define HOST_NAME_MAX 256

static int getProcessorID(cpu_set_t *cpu_set) {
    int processorId;

    for (processorId = 0; processorId < MAX_NUM_THREADS; processorId++) {
        if (CPU_ISSET(processorId, cpu_set)) {
            break;
        }
    }
    return processorId;
}

int affinity_getProcessorId(void) {
    cpu_set_t cpu_set;
    CPU_ZERO(&cpu_set);
#ifdef _OPENMP
    sched_getaffinity(gettid(), sizeof(cpu_set_t), &cpu_set);
#else
    sched_getaffinity(getpid(), sizeof(cpu_set_t), &cpu_set);
#endif

    return getProcessorID(&cpu_set);
}

void affinity_getmask(void) {
    int i = 0;
    cpu_set_t my_set;
    int nproc = sysconf(_SC_NPROCESSORS_ONLN);
    CPU_ZERO(&my_set);
    sched_getaffinity(gettid(), sizeof(cpu_set_t), &my_set);
    for (i = 0; i < nproc; i++) {
        printf("%02d ", i);
    }
    printf("\n");
    for (i = 0; i < nproc; i++) {
        if (CPU_ISSET(i, &my_set)) {
            printf(" * ");
        } else {
            printf(" - ");
        }
    }
    printf("\n");
}

int main(void) {
    int rank = 0, size = 0;
    char hostname[HOST_NAME_MAX];

    gethostname(hostname, HOST_NAME_MAX);
    pid_t master_pid = getpid();
    sleep(5);

#ifdef _OPENMP
#pragma omp parallel
    {
#pragma omp single
        printf("OpenMP enabled using %d threads\n", omp_get_num_threads());
    }
#endif
    printf("Process running on Node %s with pid %d\n", hostname, master_pid);

#ifdef _OPENMP
#pragma omp parallel
    {
#pragma omp critical
        {
            printf("Thread %d running on Node %s core %d with pid %d and "
                    "tid %d\n",
                    omp_get_thread_num(), hostname, sched_getcpu(), master_pid,
                    gettid());
            affinity_getmask();
        }
    }
#endif
    fflush(stdout);
}
